#include <string.h>
#include <stdlib.h>
#include "syna_tcm2_runtime.h"
#include "syna_tcm2_mtouch.h"

static int mtouch_screen_set_display(screen_window_t screen_win, int display_id)
{
    int i;
    int ndisplays = 0;
    int rc;
    screen_context_t screen_ctx = NULL;
    screen_display_t *screen_disp;

    rc = screen_get_window_property_pv(screen_win, SCREEN_PROPERTY_CONTEXT, (void **)&screen_ctx);
    if (rc) {
        LOGE("get display count failed ret=%d, errno=%d", rc, errno);
        error_memory("Synaptics_Touch: get display count failed ret=%d, errno=%d", rc, errno);
        return rc;
    }

    rc = screen_get_context_property_iv(screen_ctx, SCREEN_PROPERTY_DISPLAY_COUNT, &ndisplays);
    if (rc) {
        LOGE("get display count failed ret=%d, errno=%d", rc, errno);
        error_memory("Synaptics_Touch: get display count failed ret=%d, errno=%d", rc, errno);
        return rc;
    }

    screen_disp = calloc(ndisplays, sizeof(*screen_disp));
    if (screen_disp == NULL) {
        LOGE("could not allocate memory for display list, errno=%d", errno);
        error_memory("Synaptics_Touch: could not allocate memory for display list, errno=%d", errno);
        return -1;
    }

    rc = screen_get_context_property_pv(screen_ctx, SCREEN_PROPERTY_DISPLAYS, (void **)screen_disp);
    if (rc) {
        LOGE( "get display properties failed ret=%d, errno=%d", rc, errno);
        error_memory("Synaptics_Touch: get display properties failed ret=%d, errno=%d", rc, errno);
        free(screen_disp);
        return rc;
    }

    int want_id = display_id;
    for (i = 0; i < ndisplays; ++i) {
        int actual_id = 0;  // invalid
        if ((screen_get_display_property_iv(screen_disp[i], SCREEN_PROPERTY_ID, &actual_id)) == 0) {
            if (want_id == actual_id) {
                break;
            }
        }
    }

    if (i >= ndisplays) {
        LOGE( "couldn't find display %d\n", display_id);
        error_memory("Synaptics_Touch: couldn't find display %d\n", display_id);
        free(screen_disp);
        return -1;
    } else {
        rc = screen_set_window_property_pv(screen_win, SCREEN_PROPERTY_DISPLAY, (void**)&screen_disp[i]);
        if (rc) {
            LOGE("failed to set window display property, errno=%d", errno);
            error_memory("Synaptics_Touch: failed to set window display property, errno=%d", errno);
            free(screen_disp);
            return -1;
        }
    }
    free(screen_disp);
    return 0;
}

int mtouch_create_qvm_window(struct syna_tcm_dev *dev)
{
    int disp = 0;
    int size[2] = { 0, 0 };
    int pos[2] = { 0, 0 };
    int sensitivity = SCREEN_SENSITIVITY_ALWAYS;
    int transparency = SCREEN_TRANSPARENCY_SOURCE_OVER;
    int export = 1;
    int ontop = 1000;
    int visible = 1;
    int usage = SCREEN_USAGE_READ | SCREEN_USAGE_WRITE | SCREEN_USAGE_NATIVE;
    int rc = -1;
    int alpha = 0x1;
    int format = SCREEN_FORMAT_RGBA8888;
    int color = 0x00FFFFFF;

    (void) disp;

    size[0] = dev->win_width;
    size[1] = dev->win_height;

    pos[0] = dev->win_xpos;
    pos[1] = dev->win_ypos;

    rc = screen_create_context(&dev->screen_ctx, SCREEN_APPLICATION_CONTEXT);
    if (rc) {
        LOGE("failed to create screen context, ret=%d, errno=%d",
                     rc, errno);
        error_memory("Synaptics_Touch: failed to create screen context, ret=%d, errno=%d",
                     rc, errno);
        goto fail;
    }

    rc = screen_create_window_type(&dev->window, dev->screen_ctx, SCREEN_APPLICATION_WINDOW);
    if (rc) {
        LOGE("failed to create window type ret=%d, errno=%d",
                     rc, errno);
        error_memory("Synaptics_Touch: failed to create window type ret=%d, errno=%d",
                     rc, errno);
        goto fail;
    }

    rc = screen_set_window_property_iv(dev->window, SCREEN_PROPERTY_USAGE, &usage);
    if (rc) {
        LOGE("failed to set screen usage ret=%d, errno=%d",
                     rc, errno);
        error_memory("Synaptics_Touch: failed to set screen usage ret=%d, errno=%d",
                     rc, errno);
        
        goto fail;
    }

    rc = screen_set_window_property_cv(dev->window, SCREEN_PROPERTY_ID_STRING, strlen(dev->win_name), dev->win_name);
    if (rc) {
        LOGE("failed to set window name ret=%d, errno=%d",
                     rc, errno);
        error_memory("Synaptics_Touch: failed to set window name ret=%d, errno=%d",
                     rc, errno);
        goto fail;
    }

    if (pos[0] != 0 || pos[1] != 0) {
        rc = screen_set_window_property_iv(dev->window, SCREEN_PROPERTY_POSITION, pos);
    }
    if (size[0] != 0 && size[1] != 0) {
        rc = screen_set_window_property_iv(dev->window, SCREEN_PROPERTY_SIZE, size);
    }
    if (rc) {
        LOGE("failed to set window size and pos ret=%d, errno=%d",
                     rc, errno);
        error_memory("Synaptics_Touch: failed to set window size and pos ret=%d, errno=%d",
                     rc, errno);
        goto fail;
    }

    rc = screen_set_window_property_iv(dev->window, SCREEN_PROPERTY_ZORDER, &ontop);
    rc |= screen_set_window_property_iv(dev->window, SCREEN_PROPERTY_SENSITIVITY, &sensitivity);
    rc |= screen_set_window_property_iv(dev->window, SCREEN_PROPERTY_VISIBLE, &visible);
    rc |= screen_set_window_property_iv(dev->window, SCREEN_PROPERTY_TRANSPARENCY, &transparency);
    rc |= screen_set_window_property_iv(dev->window, SCREEN_PROPERTY_FORMAT , &format);
    rc |= screen_set_window_property_iv(dev->window, SCREEN_PROPERTY_COLOR , &color);
    rc |= screen_set_window_property_iv(dev->window, SCREEN_PROPERTY_GLOBAL_ALPHA, &alpha);
    if (rc < 0) {
        LOGE("failed to set window properties ret=%d, errno=%d",
                     rc, errno);
        error_memory("Synaptics_Touch: failed to set window properties ret=%d, errno=%d",
                     rc, errno);
        goto fail;
    }

    if (dev->disp_id) {
        if ((rc = mtouch_screen_set_display(dev->window, dev->disp_id)) != 0) {
            LOGE("failed to set screen display ret=%d", rc);
            error_memory("Synaptics_Touch: failed to set screen display ret=%d", rc);
            goto fail;
        }
    }
    else {
        rc = -1;
        LOGE("no display id provided");
        error_memory("Synaptics_Touch: no display id provided");
        goto fail;
    }

    rc = screen_set_window_property_iv(dev->window, SCREEN_PROPERTY_BUFFER_SIZE, size);
    if (rc)
    {
        LOGE("screen_set_window_property_iv of SCREEN_PROPERTY_BUFFER_SIZE failed, rc =%d, errno =%d",
                     rc, errno);
        error_memory("Synaptics_Touch: screen_set_window_property_iv of SCREEN_PROPERTY_BUFFER_SIZE failed, rc =%d, errno =%d",
                     rc, errno);
        goto fail;
    }

    rc = screen_create_window_buffers(dev->window, 0);
    if (rc) {
        LOGE("create buffer failed ret=%d, errno=%d", rc, errno);
        error_memory("Synaptics_Touch: create buffer failed ret=%d, errno=%d", rc, errno);
        goto fail;
    }

    /* Target the permission with the pid process:1234:rwx */
    char permissions[] = "other::rwx";
    //int permission = SCREEN_PERMISSION_IROTH | SCREEN_PERMISSION_IWOTH | SCREEN_PERMISSION_IXOTH;
    if(export) {
        rc = screen_set_window_property_cv(dev->window, SCREEN_PROPERTY_PERMISSIONS, strlen(permissions), permissions);
    }

    rc |= screen_flush_context(dev->screen_ctx, 0);
    if (rc) {
        goto fail;
    }

    screen_buffer_t buffer = NULL;
    rc = screen_get_window_property_pv(dev->window, SCREEN_PROPERTY_RENDER_BUFFERS, (void **)&buffer);
    if (rc) {
        LOGE("failed to render buffer ret=%d, errno=%d", rc, errno);
        error_memory("Synaptics_Touch: failed to render buffer ret=%d, errno=%d", rc, errno);
        goto fail;
    }

    return rc;

fail:
    if (dev->window) {
        screen_destroy_window(dev->window);
    }
    if (dev->screen_ctx) {
        screen_destroy_context(dev->screen_ctx);
    }
    return rc;
}
